﻿Imports System.Runtime.InteropServices

Public Class SHPDLL
    '.NET Framework wrapper for Shapefile C Library V1.2.10
    'Shapefile C Library is (c) 1998 Frank Warmerdam.  .NET wrapper provided by David Gancarz.
    'Please send error reports or other suggestions regarding this wrapper class to:
    'dgancarz@cfl.rr.com or david.gancarz@cityoforlando.net

    Public Enum ShapeType 'Shape type enumeration
        NullShape = 0 'Shape with no geometric data
        Point = 1 '2D point
        PolyLine = 3 '2D polyline
        Polygon = 5 '2D polygon
        MultiPoint = 8 'Set of 2D points
        PointZ = 11 '3D point
        PolyLineZ = 13 '3D polyline
        PolygonZ = 15 '3D polygon
        MultiPointZ = 18 'Set of 3D points
        PointM = 21 '3D point with measure
        PolyLineM = 23 '3D polyline with measure
        PolygonM = 25 '3D polygon with measure
        MultiPointM = 28 'Set of 3d points with measures
        MultiPatch = 31 'Collection of surface patches
    End Enum

    Public Enum PartType 'Part type enumeration - everything but ShapeType.MultiPatch just uses PartType.Ring.
        TriangleStrip = 0 'Linked strip of triangles. Every vertex (after the first two) completes a new triangle.
        TriangleFan = 1 'A linked fan of triangles. A new triangle is always formed by connecting the new vertex with its immediate predecessor and the first vertex of the part.
        OuterRing = 2 'The outer ring of a polygon
        InnerRing = 3 'The first ring of a polygon
        FirstRing = 4 'The outer ring of a polygon of an unspecified type
        Ring = 5 'A ring of a polygon of an unspecified type
    End Enum

    <StructLayout(LayoutKind.Sequential)> _
    Public Class SHPObject
        Public shpType As ShapeType 'Shape type as a ShapeType enum
        Public nShapeId As Integer 'Shape number (-1 is unknown/unassigned)
        Public nParts As Integer 'Number of parts (0 implies single part with no info)
        Public paPartStart As IntPtr 'Pointer to int array of part start offsets, of size nParts
        Public paPartType As IntPtr 'Pointer to PartType array (PartType.Ring if not ShapeType.MultiPatch) of size nParts
        Public nVertices As Integer 'Number of vertices
        Public padfX As IntPtr 'Pointer to double array containing X coordinates
        Public padfY As IntPtr 'Pointer to double array containing Y coordinates
        Public padfZ As IntPtr 'Pointer to double array containing Z coordinates (all zero if not provided)
        Public padfM As IntPtr 'Pointer to double array containing Measure coordinates(all zero if not provided)
        Public dfXMin As Double 'Bounding rectangle's min X
        Public dfYMin As Double 'Bounding rectangle's min Y
        Public dfZMin As Double 'Bounding rectangle's min Z
        Public dfMMin As Double 'Bounding rectangle's min M
        Public dfXMax As Double 'Bounding rectangle's max X
        Public dfYMax As Double 'Bounding rectangle's max Y
        Public dfZMax As Double 'Bounding rectangle's max Z
        Public dfMMax As Double 'Bounding rectangle's max M
    End Class


    'The SHPOpen() function should be used to establish access to the two files for 
    'accessing vertices (.shp and .shx). Note that both files have to be in the indicated 
    'directory, and must have the expected extensions in lower case. The returned SHPHandle 
    'is passed to other access functions, and SHPClose() should be invoked to recover 
    'resources, and flush changes to disk when complete.
    'szShapeFile = The name of the layer can be the name of either the .shp or the .shx file or can just be the path plus the basename of the pair.
    'szAccess = At this time only "rb" (read-only binary) and "rb+" (read/write binary) should be used.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPOpen(ByVal szShapeFile As String, ByVal szAccess As String) As IntPtr
    End Function

    'The SHPCreate() function will create a new .shp and .shx file of the desired type.
    'szShapeFile = The name of the layer to access. This can be the name of either the .shp or the .shx file or can just be the path plus the basename of the pair.
    'shpType = ShapeType.Point, ShapeType.PolyLine, ShapeType.Polygon or ShapeType.MultiPoint.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPCreate(ByVal szShapeFile As String, ByVal shpType As ShapeType) As IntPtr
    End Function

    'The SHPGetInfo() function retrieves various information about shapefile as a whole.
    'The bounds are read from the file header, and may be inaccurate if the file was improperly generated.
    'hSHP = The handle previously returned by SHPOpen() or SHPCreate()
    'pnEntities = A pointer to an integer into which the number of entities/structures should be placed. May be NULL.
    'pshpType = A pointer to an integer into which the ShapeType of this file should be placed. Shapefiles may contain either ShapeType.Point, ShapeType.PolyLine, ShapeType.Polygon or ShapeType.MultiPoint entities. This may be NULL.
    'adfMinBound = The X, Y, Z and M minimum values will be placed into this four entry array. This may be NULL.
    'adfMaxBound = The X, Y, Z and M maximum values will be placed into this four entry array. This may be NULL.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPGetInfo(ByVal hSHP As IntPtr, ByRef pnEntities As Integer, ByRef pshpType As ShapeType, ByVal adfMinBound() As Double, ByVal adfMaxBound() As Double)
    End Sub

    'The SHPReadObject() call is used to read a single structure, or entity from the shapefile.
    'See the definition of the SHPObject structure for detailed information on fields of a SHPObject.
    'hSHP = The handle previously returned by SHPOpen() or SHPCreate().
    'iShape = The entity number of the shape to read. Entity numbers are between 0 and nEntities-1 (as returned by SHPGetInfo()).
    'Returns IntPtr to a SHPObject.  Use System.Runtime.InteropServices.Marshal.PtrToStructure to
    'create a SHPObject copy in managed memory.
    'IntPtr's returned from SHPReadObject() should be deallocated with SHPDestroyShape(). 
    'SHPReadObject() will return IntPtr.Zero if an illegal iShape value is requested. 
    'Note that the bounds placed into the SHPObject are those read from the file, and may not be correct. 
    'For points the bounds are generated from the single point since bounds aren't normally provided 
    'for point types. Generally the shapes returned will be of the type of the file as a whole. 
    'However, any file may also contain type ShapeType.NullShape shapes which will have no geometry. 
    'Generally speaking applications should skip rather than preserve them, as they usually 
    'represented interactively deleted shapes.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPReadObject(ByVal hSHP As IntPtr, ByVal iShape As Integer) As IntPtr
    End Function

    'The SHPWriteObject() call is used to write a single structure, or entity to the shapefile. 
    'See the definition of the SHPObject structure for detailed information on fields of a SHPObject.
    'hSHP = The handle previously returned by SHPOpen("r+") or SHPCreate().
    'iShape = The entity number of the shape to write. A value of -1 should be used for new shapes.
    'psObject = The shape to write to the file. This should have been created with SHPCreateObject(), or SHPCreateSimpleObject().
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPWriteObject(ByVal hSHP As IntPtr, ByVal iShape As Integer, ByVal psObject As IntPtr) As Integer
    End Function

    'This function should be used to deallocate the resources associated with a SHPObject 
    'when it is no longer needed, including those created with SHPCreateSimpleObject(), 
    'SHPCreateObject() and returned from SHPReadObject().
    'psObject = IntPtr of the SHPObject to deallocate memory.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPDestroyObject(ByVal psObject As IntPtr)
    End Sub

    'This function will recompute the extents of this shape, replacing the existing values 
    'of the dfXMin, dfYMin, dfZMin, dfMMin, dfXMax, dfYMax, dfZMax, and dfMMax values based 
    'on the current set of vertices for the shape. This function is automatically called by 
    'SHPCreateObject() but if the vertices of an existing object are altered it should be 
    'called again to fix up the extents.
    'psObject = An existing shape object to be updated in place.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPComputeExtents(ByVal psObject As IntPtr)
    End Sub

    'The SHPCreateObject() function allows for the creation of objects (shapes). 
    'This is normally used so that the SHPObject can be passed to SHPWriteObject() 
    'to write it to the file.
    'shpType = The ShapeType of the object to be created, such as ShapeType.Point, or ShapeType.Polygon.
    'nShapeId = The shapeid to be recorded with this shape.
    'nParts = The number of parts for this object. If this is zero for PolyLine, or Polygon type objects, a single zero valued part will be created internally.
    'panPartStart = The list of zero based start vertices for the rings (parts) in this object. The first should always be zero. This may be NULL if nParts is 0.
    'paPartType = The type of each of the parts. This is only meaningful for MultiPatch files. For all other cases this may be NULL, and will be assumed to be PartType.Ring.
    'nVertices = The number of vertices being passed in padfX, padfY, and padfZ. 
    'adfX = An array of nVertices X coordinates of the vertices for this object.
    'adfY = An array of nVertices Y coordinates of the vertices for this object.
    'adfZ = An array of nVertices Z coordinates of the vertices for this object. This may be NULL in which case they are all assumed to be zero.
    'adfM = An array of nVertices M (measure values) of the vertices for this object. This may be NULL in which case they are all assumed to be zero.
    'The SHPDestroyObject() function should be used to free resources associated with an object allocated with SHPCreateObject(). This function computes a bounding box for the SHPObject from the given vertices.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPCreateObject(ByVal shpType As ShapeType, ByVal nShapeId As Integer, ByVal nParts As Integer, ByVal panPartStart() As Integer, ByVal paPartType() As PartType, ByVal nVertices As Integer, ByVal adfX() As Double, ByVal adfY() As Double, ByVal adfZ() As Double, ByVal adfM() As Double) As IntPtr
    End Function

    'The SHPCreateSimpleObject() function allows for the convenient creation of simple objects. 
    'This is normally used so that the SHPObject can be passed to SHPWriteObject() to write it 
    'to the file. The simple object creation API assumes an M (measure) value of zero for each vertex. 
    'For complex objects (such as polygons) it is assumed that there is only one part, and that it 
    'is of the default type (PartType.Ring). Use the SHPCreateObject() function for more sophisticated objects. 
    'shpType = The ShapeType of the object to be created, such as ShapeType.Point, or ShapeType.Polygon.
    'nVertices = The number of vertices being passed in padfX, padfY, and padfZ.
    'adfX = An array of nVertices X coordinates of the vertices for this object.
    'adfY = An array of nVertices Y coordinates of the vertices for this object.
    'adfZ = An array of nVertices Z coordinates of the vertices for this object. This may be NULL in which case they are all assumed to be zero.
    'The SHPDestroyObject() function should be used to free resources associated with an object allocated with SHPCreateSimpleObject().
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPCreateSimpleObject(ByVal shpType As ShapeType, ByVal nVertices As Integer, ByVal adfX() As Double, ByVal adfY() As Double, ByVal adfZ() As Double) As IntPtr
    End Function

    'The SHPClose() function will close the .shp and .shx files, and flush all outstanding header 
    'information to the files. It will also recover resources associated with the handle. 
    'After this call the hSHP handle cannot be used again.
    'hSHP = The handle previously returned by SHPOpen() or SHPCreate().
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPClose(ByVal hSHP As IntPtr)
    End Sub

    'Translates a ShapeType.* constant into a named shape type (Point, PointZ, Polygon, etc.)
    'shpType = ShapeType enum
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPTypeName(ByVal shpType As ShapeType) As String
    End Function

    'Translates a PartType enum into a named part type (Ring, Inner Ring, etc.)
    'partType = PartType enum
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPPartTypeName(ByVal partType As PartType) As String
    End Function

    ' -------------------------------------------------------------------- 
    '      Shape quadtree indexing API.                                    
    ' -------------------------------------------------------------------- 
    'Creates a quadtree index
    'hSHP = ?
    'nDimension = ?
    'nMaxDepth = ?
    'adfBoundsMin = ?
    'adfBoundsMax = ?
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPCreateTree(ByVal hSHP As IntPtr, ByVal nDimension As Integer, ByVal nMaxDepth As Integer, ByVal adfBoundsMin() As Double, ByVal adfBoundsMax() As Double) As IntPtr
    End Function

    'Releases resources associated with quadtree
    'hTree = ?
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPDestroyTree(ByVal hTree As IntPtr)
    End Sub

    'hTree = ?
    'psObject = ?
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPTreeAddShapeId(ByVal hTree As IntPtr, ByVal psObject As IntPtr) As Integer
    End Function

    'No Description
    'hTree = ?
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub SHPTreeTrimExtraNodes(ByVal hTree As IntPtr)
    End Sub

    'No Description
    'hTree">
    'adfBoundsMin">
    'adfBoundsMax">
    'pnShapeCount">
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPTreeFindLikelyShapes(ByVal hTree As IntPtr, ByVal adfBoundsMin() As Double, ByVal adfBoundsMax() As Double, ByRef pnShapeCount As Integer) As IntPtr
    End Function

    'No Description
    'adfBox1Min">
    'adfBox1Max">
    'adfBox2Min">
    'adfBox2Max">
    'nDimension">
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function SHPCheckBoundsOverlap(ByVal adfBox1Min() As Double, ByVal adfBox1Max() As Double, ByVal adfBox2Min() As Double, ByVal adfBox2Max() As Double, ByVal nDimension As Integer) As Integer
    End Function

    'xBase field type enumeration
    Public Enum DBFFieldType
        FTString 'String data type
        FTInteger 'Integer data type
        FTDouble 'Double data type
        FTLogical 'Logical data type
        FTInvalid 'Invalid data type
        FTDate 'Date data type
    End Enum

    'The DBFOpen() function should be used to establish access to an existing xBase format table file. 
    'The returned DBFHandle is passed to other access functions, and DBFClose() should be invoked 
    'to recover resources, and flush changes to disk when complete. The DBFCreate() function should 
    'called to create new xBase files. As a convenience, DBFOpen() can be called with the name of a 
    '.shp or .shx file, and it will figure out the name of the related .dbf file.
    'szDBFFile = The name of the xBase (.dbf) file to access.
    'szAccess = The fopen() style access string. At this time only "rb" (read-only binary) and "rb+" (read/write binary) should be used.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFOpen(ByVal szDBFFile As String, ByVal szAccess As String) As IntPtr
    End Function

    'The DBFCreate() function creates a new xBase format file with the given name, 
    'and returns an access handle that can be used with other DBF functions. 
    'The newly created file will have no fields, and no records. 
    'Fields should be added with DBFAddField() before any records add written. 
    'szDBFFile = The name of the xBase (.dbf) file to create.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFCreate(ByVal szDBFFile As String) As IntPtr
    End Function

    'The DBFGetFieldCount() function returns the number of fields currently defined for the indicated xBase file. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFGetFieldCount(ByVal hDBF As IntPtr) As Integer
    End Function

    'The DBFGetRecordCount() function returns the number of records that exist on the xBase 
    'file currently. Note that with shape files one xBase record exists for each shape in the .shp/.shx files.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFGetRecordCount(ByVal hDBF As IntPtr) As Integer
    End Function

    'The DBFAddField() function is used to add new fields to an existing xBase file opened with DBFOpen(), 
    'or created with DBFCreate(). Note that fields can only be added to xBase files with no records, 
    'though this is limitation of this API, not of the file format. Returns the field number of the 
    'new field, or -1 if the addition of the field failed
    'hDBF = The access handle for the file to be updated, as returned by DBFOpen(), or DBFCreate().
    'szFieldName = The name of the new field. At most 11 character will be used. 
    'In order to use the xBase file in some packages it may be necessary to avoid some special 
    'characters in the field names such as spaces, or arithmetic operators.
    'eType = One of FTString, FTInteger, FTLogical, FTDate, or FTDouble in order to establish the 
    'type of the new field. Note that some valid xBase field types cannot be created such as date fields.
    'nWidth = The width of the field to be created. For FTString fields this establishes 
    'the maximum length of string that can be stored. For FTInteger this establishes the number of 
    'digits of the largest number that can be represented. For FTDouble fields this in combination 
    'with the nDecimals value establish the size, and precision of the created field.
    'nDecimals = The number of decimal places to reserve for FTDouble fields. 
    'For all other field types this should be zero. For instance with nWidth=7, and nDecimals=3 
    'numbers would be formatted similarly to `123.456'.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFAddField(ByVal hDBF As IntPtr, ByVal szFieldName As String, ByVal eType As DBFFieldType, ByVal nWidth As Integer, ByVal nDecimals As Integer) As Integer
    End Function

    'The DBFGetFieldInfo() returns the type of the requested field, which is one of the DBFFieldType 
    'enumerated values. As well, the field name, and field width information can optionally be returned. 
    'The field type returned does not correspond one to one with the xBase field types. 
    'For instance the xBase field type for Date will just be returned as being FTInteger. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iField = The field to be queried. This should be a number between 0 and n-1, 
    'where n is the number fields on the file, as returned by DBFGetFieldCount().
    'szFieldName = If this pointer is not NULL the name of the requested field 
    'will be written to this location. The pszFieldName buffer should be at least 12 character 
    'is size in order to hold the longest possible field name of 11 characters plus a terminating 
    'zero character.
    'pnWidth = If this pointer is not NULL, the width of the requested field will be 
    'returned in the int pointed to by pnWidth. This is the width in characters. 
    'pnDecimals = If this pointer is not NULL, the number of decimal places precision 
    'defined for the field will be returned. This is zero for integer fields, or non-numeric fields.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFGetFieldInfo(ByVal hDBF As IntPtr, ByVal iField As Integer, ByVal szFieldName As System.Text.StringBuilder, ByRef pnWidth As Integer, ByRef pnDecimals As Integer) As DBFFieldType
    End Function

    'Returns the index of the field matching this name, or -1 on failure. 
    'The comparison is case insensitive. However, lengths must match exactly.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'szFieldName = Name of the field to search for.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFGetFieldIndex(ByVal hDBF As IntPtr, ByVal szFieldName As String) As Integer
    End Function

    'The DBFReadIntegerAttribute() will read the value of one field and return it as an integer. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'This can be used even with FTString fields, though the returned value will be zero if not interpretable as a number.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFReadIntegerAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Integer
    End Function

    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi, EntryPoint:="DBFReadDateAttribute")> _
    Private Shared Function _DBFReadDateAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Integer
    End Function

    'The DBFReadDateAttribute() will read the value of one field and return it as a System.DateTime value.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'This can be used even with FTString fields, though the returned value will be zero if not interpretable as a number.
    Public Shared Function DBFReadDateAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As DateTime
        Dim sDate As String = _DBFReadDateAttribute(hDBF, iShape, iField).ToString()
        Try
            Dim d As New DateTime(Integer.Parse(sDate.Substring(0, 4)), Integer.Parse(sDate.Substring(4, 2)), Integer.Parse(sDate.Substring(6, 2)))
            Return d
        Catch
            Return New DateTime(0)
        End Try
    End Function

    'The DBFReadDoubleAttribute() will read the value of one field and return it as a double. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'This can be used even with FTString fields, though the returned value will be zero if not interpretable as a number.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFReadDoubleAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Double
    End Function

    'The DBFReadStringAttribute() will read the value of one field and return it as a string. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'This function may be used on any field type (including FTInteger and FTDouble) and will 
    'return the string representation stored in the .dbf file. The returned pointer is to an 
    'internal buffer which is only valid untill the next DBF function call. It's contents may 
    'be copied with normal string functions such as strcpy(), or strdup(). If the 
    'TRIM_DBF_WHITESPACE macro is defined in shapefil.h (it is by default) then all leading and 
    'trailing space (ASCII 32) characters will be stripped before the string is returned.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFReadStringAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As String
    End Function

    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi, EntryPoint:="DBFReadLogicalAttribute")> _
    Private Shared Function _DBFReadLogicalAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As String
    End Function

    'The DBFReadLogicalAttribute() will read the value of one field and return it as a boolean. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'This can be used with FTString fields, in which case it returns TRUE if the string="T"; otherwise it returns FALSE.
    Public Shared Function DBFReadLogicalAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Boolean
        Return (_DBFReadLogicalAttribute(hDBF, iShape, iField) = "T")
    End Function

    'This function will return TRUE if the indicated field is NULL valued otherwise FALSE. 
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) from which the field value should be read.
    'iField = The field within the selected record that should be read.
    'Note that NULL fields are represented in the .dbf file as having all spaces in the field. 
    'Reading NULL fields will result in a value of 0.0 or an empty string with the other 
    'DBFRead*Attribute() functions.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFIsAttributeNULL(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Integer
    End Function

    'The DBFWriteIntegerAttribute() function is used to write a value to a numeric field 
    '(FTInteger, or FTDouble). If the write succeeds the value TRUE will be returned, 
    'otherwise FALSE will be returned. If the value is too large to fit in the field, 
    'it will be truncated and FALSE returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'nFieldValue = The integer value that should be written.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteIntegerAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal nFieldValue As Integer) As Integer
    End Function

    'Write an attribute record to the file, but without any reformatting based on type.  
    'The provided buffer is written as is to the field position in the record. 
    'If the write succeeds the value TRUE will be returned, otherwise FALSE will be returned. 
    'If the value is too large to fit in the field, it will be truncated and FALSE returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'pValue = pointer to the value
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteAttributeDirectly(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal pValue As IntPtr) As Integer
    End Function

    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi, EntryPoint:="DBFWriteDateAttribute")> _
    Private Shared Function _DBFWriteDateAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal nFieldValue As Integer) As Integer
    End Function

    'The DBFWriteDateAttribute() function is used to write a value to a date field 
    '(FTDate). If the write succeeds the value TRUE will be returned, 
    'otherwise FALSE will be returned. If the value is too large to fit in the field, 
    'it will be truncated and FALSE returned. 
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'nFieldValue = The System.DateTime date value that should be written.
    Public Shared Function DBFWriteDateAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal nFieldValue As DateTime) As Integer
        Return _DBFWriteDateAttribute(hDBF, iShape, iField, Integer.Parse(nFieldValue.ToString("yyyyMMdd")))
    End Function

    'The DBFWriteDoubleAttribute() function is used to write a value to a numeric field 
    '(FTInteger, or FTDouble). If the write succeeds the value TRUE will be returned, 
    'otherwise FALSE will be returned. If the value is too large to fit in the field, 
    'it will be truncated and FALSE returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'dFieldValue = The floating point value that should be written.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteDoubleAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal dFieldValue As Double) As Integer
    End Function

    'The DBFWriteStringAttribute() function is used to write a value to a string field (FString). 
    'If the write succeeds the value TRUE willbe returned, otherwise FALSE will be returned. 
    'If the value is too large to fit in the field, it will be truncated and FALSE returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'szFieldValue = The string to be written to the field.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteStringAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal szFieldValue As String) As Integer
    End Function

    'The DBFWriteNULLAttribute() function is used to clear the indicated field to a NULL value.
    'In the .dbf file this is represented by setting the entire field to spaces. If the write
    'succeeds the value TRUE will be returned, otherwise FALSE will be returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteNULLAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer) As Integer
    End Function

    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi, EntryPoint:="DBFWriteLogicalAttribute")> _
    Private Shared Function _DBFWriteLogicalAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal lFieldValue As Char) As Integer
    End Function

    'The DBFWriteLogicalAttribute() function is used to write a boolean value to a logical field 
    '(FTLogical). If the write succeeds the value TRUE will be returned, otherwise FALSE will be returned.
    'hDBF = The access handle for the file to be written, as returned by DBFOpen(), or DBFCreate().
    'iShape = The record number (shape number) to which the field value should be written.
    'iField = The field within the selected record that should be written.
    'bFieldValue = The boolean value to be written to the field.
    Public Shared Function DBFWriteLogicalAttribute(ByVal hDBF As IntPtr, ByVal iShape As Integer, ByVal iField As Integer, ByVal bFieldValue As Boolean) As Integer
        If bFieldValue Then
            Return _DBFWriteLogicalAttribute(hDBF, iShape, iField, "T"c)
        Else
            Return _DBFWriteLogicalAttribute(hDBF, iShape, iField, "F"c)
        End If
    End Function

    'Reads the attribute fields of a record.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'hEntity = The entity (record) number to be read
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFReadTuple(ByVal hDBF As IntPtr, ByVal hEntity As Integer) As IntPtr
    End Function

    'Writes an attribute record to the file.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'hEntity = The zero-based entity (record) number to be written.  If hEntity equals the number of records a new record is appended.
    'pRawTuple = Pointer to the tuple to be written
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFWriteTuple(ByVal hDBF As IntPtr, ByVal hEntity As Integer, ByVal pRawTuple As IntPtr) As Integer
    End Function

    'Copies the data structure of an xBase file to another xBase file.  
    'Data are not copied.  Use Read/WriteTuple functions to selectively copy data.
    'hDBF = The access handle for the file to be queried, as returned by DBFOpen(), or DBFCreate().
    'szFilename = The name of the xBase (.dbf) file to create.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFCloneEmpty(ByVal hDBF As IntPtr, ByVal szFilename As String) As IntPtr
    End Function

    'The DBFClose() function will close the indicated xBase file (opened with DBFOpen(), 
    'or DBFCreate()), flushing out all information to the file on disk, and recovering 
    'any resources associated with having the file open. The file handle (hDBF) should not 
    'be used again with the DBF API after calling DBFClose().
    'hDBF = The access handle for the file to be closed.
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Sub DBFClose(ByVal hDBF As IntPtr)
    End Sub

    'This function returns the DBF type code of the indicated field.
    'hDBF = The access handle for the file.
    'iField = The field index to query.
    'Return value will be one of:
    'C: String
    'D: Date
    'F: Float
    'N: Numeric, with or without decimal
    'L: Logical
    'M: Memo: 10 digits .DBT block ptr
    ' :field out of range
    <DllImport("shapelib.dll", CharSet:=CharSet.Ansi)> _
    Public Shared Function DBFGetNativeFieldType(ByVal hDBF As IntPtr, ByVal iField As Integer) As SByte
    End Function
End Class
